require('dotenv').config();
const express = require('express');
const jwt = require('jsonwebtoken');
const cors = require('cors');
const helmet = require('helmet');
const path = require('path');
const Database = require('better-sqlite3');
const crypto = require('crypto');

const app = express();
app.use(helmet());
app.use(express.json());
app.use(express.urlencoded({ extended: true }));

const JWT_SECRET = process.env.JWT_SECRET;
const ADMIN_KEY = process.env.ADMIN_KEY;
if (!JWT_SECRET) {
  console.error('Error: JWT_SECRET not set in env (see .env.example)');
  process.exit(1);
}
if (!ADMIN_KEY) {
  console.error('Error: ADMIN_KEY not set in env (see .env.example)');
  process.exit(1);
}

// DB setup
const dataDir = path.join(__dirname, 'data');
const dbPath = path.join(dataDir, 'licenses.db');
const fs = require('fs');
if (!fs.existsSync(dataDir)) fs.mkdirSync(dataDir, { recursive: true });
const db = new Database(dbPath);
db.prepare(`CREATE TABLE IF NOT EXISTS licenses (
  id INTEGER PRIMARY KEY AUTOINCREMENT,
  domain TEXT NOT NULL UNIQUE,
  active INTEGER NOT NULL DEFAULT 1,
  client_key TEXT,
  cpanel_url TEXT,
  cpanel_user TEXT,
  cpanel_token_enc TEXT,
  created_at INTEGER NOT NULL,
  updated_at INTEGER NOT NULL,
  note TEXT
)`).run();

const ALLOWED_ORIGINS = (process.env.ALLOWED_ORIGINS || '').split(',').filter(Boolean);

// CORS
app.use(cors({
  origin: function(origin, callback){
    if (!origin) return callback(new Error('Origin not allowed'), false);
    if (ALLOWED_ORIGINS.length === 0) {
      return callback(new Error('No allowed origins configured on server'), false);
    }
    if (ALLOWED_ORIGINS.some(o => origin.startsWith(o))) return callback(null, true);
    callback(new Error('Origin not allowed'), false);
  },
  credentials: true
}));

// Serve admin at root and other static pages
app.use('/', express.static(path.join(__dirname, 'admin'))); // admin UI at root
app.use('/home', express.static(path.join(__dirname, 'home')));
app.use('/docs', express.static(path.join(__dirname, 'docs')));

app.get('/health', (req, res) => res.send('ok'));

// encryption helpers
const ALGO = 'aes-256-gcm';
function encrypt(text) {
  const key = crypto.createHash('sha256').update(ADMIN_KEY).digest();
  const iv = crypto.randomBytes(12);
  const cipher = crypto.createCipheriv(ALGO, key, iv);
  const encrypted = Buffer.concat([cipher.update(text, 'utf8'), cipher.final()]);
  const tag = cipher.getAuthTag();
  return Buffer.concat([iv, tag, encrypted]).toString('base64');
}
function decrypt(enc) {
  try {
    const data = Buffer.from(enc, 'base64');
    const iv = data.slice(0,12);
    const tag = data.slice(12,28);
    const encrypted = data.slice(28);
    const key = crypto.createHash('sha256').update(ADMIN_KEY).digest();
    const decipher = require('crypto').createDecipheriv(ALGO, key, iv);
    decipher.setAuthTag(tag);
    const out = Buffer.concat([decipher.update(encrypted), decipher.final()]);
    return out.toString('utf8');
  } catch (e) {
    return null;
  }
}

// DB statements
const insertLicense = db.prepare('INSERT OR IGNORE INTO licenses (domain, active, client_key, cpanel_url, cpanel_user, cpanel_token_enc, created_at, updated_at, note) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)');
const updateActive = db.prepare('UPDATE licenses SET active = ?, updated_at = ? WHERE domain = ?');
const updateClientKey = db.prepare('UPDATE licenses SET client_key = ?, updated_at = ? WHERE domain = ?');
const selectAll = db.prepare('SELECT id,domain,active,cpanel_url,cpanel_user,created_at,updated_at,client_key,note FROM licenses ORDER BY id DESC');
const selectByDomain = db.prepare('SELECT * FROM licenses WHERE domain = ? LIMIT 1');
const updateCpanel = db.prepare('UPDATE licenses SET cpanel_url = ?, cpanel_user = ?, cpanel_token_enc = ?, updated_at = ? WHERE domain = ?');

// Admin middleware
function requireAdmin(req, res, next) {
  const key = req.get('x-admin-key') || req.body.admin_key || '';
  if (!key || key !== ADMIN_KEY) return res.status(401).json({ error: 'Unauthorized (admin key required)' });
  next();
}

// helper to generate strong client key
function genClientKey(){ return crypto.randomBytes(24).toString('hex'); }

// Admin APIs
// create/reactivate license - returns client_key (if newly created or reset) in response
app.post('/admin/api/licenses', requireAdmin, (req, res) => {
  const { domain, note, reset_client_key } = req.body || {};
  if (!domain) return res.status(400).json({ error: 'domain required' });
  const now = Math.floor(Date.now()/1000);
  let existing = selectByDomain.get(domain);
  if (!existing) {
    const client_key = genClientKey();
    insertLicense.run(domain, 1, client_key, null, null, null, now, now, note || null);
    existing = selectByDomain.get(domain);
    res.json({ license: existing, client_key });
    return;
  } else {
    // reactivate
    updateActive.run(1, now, domain);
    if (reset_client_key) {
      const client_key = genClientKey();
      updateClientKey.run(client_key, now, domain);
      existing = selectByDomain.get(domain);
      res.json({ license: existing, client_key });
      return;
    }
    existing = selectByDomain.get(domain);
    res.json({ license: existing });
    return;
  }
});

app.get('/admin/api/licenses', requireAdmin, (req, res) => {
  const rows = selectAll.all();
  res.json({ licenses: rows });
});

app.post('/admin/api/licenses/activate', requireAdmin, (req, res) => {
  const { domain } = req.body || {};
  if (!domain) return res.status(400).json({ error: 'domain required' });
  const now = Math.floor(Date.now()/1000);
  updateActive.run(1, now, domain);
  const lic = selectByDomain.get(domain);
  res.json({ license: lic });
});
app.post('/admin/api/licenses/deactivate', requireAdmin, (req, res) => {
  const { domain } = req.body || {};
  if (!domain) return res.status(400).json({ error: 'domain required' });
  const now = Math.floor(Date.now()/1000);
  updateActive.run(0, now, domain);
  const lic = selectByDomain.get(domain);
  res.json({ license: lic });
});

app.post('/admin/api/licenses/cpanel', requireAdmin, (req, res) => {
  const { domain, cpanel_url, cpanel_user, cpanel_token } = req.body || {};
  if (!domain) return res.status(400).json({ error: 'domain required' });
  const now = Math.floor(Date.now()/1000);
  const enc = cpanel_token ? encrypt(cpanel_token) : null;
  updateCpanel.run(cpanel_url || null, cpanel_user || null, enc, now, domain);
  const lic = selectByDomain.get(domain);
  res.json({ license: lic });
});

app.get('/admin/api/licenses/cpanel', requireAdmin, (req, res) => {
  const domain = req.query.domain;
  if (!domain) return res.status(400).json({ error: 'domain required' });
  const lic = selectByDomain.get(domain);
  if (!lic) return res.status(404).json({ error: 'License not found' });
  const token = lic.cpanel_token_enc ? decrypt(lic.cpanel_token_enc) : null;
  res.json({ domain: lic.domain, cpanel_url: lic.cpanel_url, cpanel_user: lic.cpanel_user, cpanel_token: token });
});

// License issuing: now requires the client to send x-client-key header that must match DB client_key for domain.
app.get('/license', (req, res) => {
  const origin = req.get('origin') || req.get('referer') || '';
  if (!ALLOWED_ORIGINS.some(o => origin.startsWith(o))) {
    return res.status(403).json({ error: 'Forbidden origin (not in ALLOWED_ORIGINS)' });
  }
  const lic = db.prepare('SELECT * FROM licenses WHERE domain = ? OR ? LIKE domain || "%" LIMIT 1').get(origin, origin);
  if (!lic) return res.status(403).json({ error: 'No license found for this origin' });
  if (!lic.active) return res.status(403).json({ error: 'License is deactivated' });
  const clientKey = req.get('x-client-key') || '';
  if (!clientKey || clientKey !== lic.client_key) {
    return res.status(401).json({ error: 'Invalid or missing client key' });
  }
  const payload = { iss: 'your-company', aud: origin, lic_id: lic.id };
  const token = jwt.sign(payload, JWT_SECRET, { expiresIn: '10m' });
  res.json({ token, expiresIn: 600 });
});

// Protected API
app.get('/api/protected-data', (req, res) => {
  const auth = (req.get('authorization') || '');
  if (!auth.startsWith('Bearer ')) return res.status(401).json({ error: 'Missing token' });
  const token = auth.slice(7);
  try {
    const payload = jwt.verify(token, JWT_SECRET);
    const lic = selectByDomain.get(payload.aud) || db.prepare('SELECT * FROM licenses WHERE id = ?').get(payload.lic_id);
    if (!lic || !lic.active) return res.status(401).json({ error: 'License revoked or inactive' });
    res.json({ secret: 'protected data', license: { id: lic.id, domain: lic.domain } });
  } catch (err) {
    return res.status(401).json({ error: 'Invalid or expired token' });
  }
});

// logs
const logsDir = path.join(__dirname,'logs');
if (!fs.existsSync(logsDir)) fs.mkdirSync(logsDir,{recursive:true});
function logAttempt(info){ fs.appendFileSync(path.join(logsDir,'attempts.log'), JSON.stringify(info) + '\n'); }

app.use('/license', (req, res, next) => {
  const origin = req.get('origin') || req.get('referer') || '';
  const ip = req.ip || req.connection.remoteAddress;
  const ua = req.get('user-agent') || '';
  logAttempt({ time: Date.now(), origin, ip, ua, path: req.originalUrl });
  next();
});

app.get('/admin/api/logs', requireAdmin, (req, res) => {
  const logsPath = path.join(__dirname, 'logs', 'attempts.log');
  if (!fs.existsSync(logsPath)) return res.json({ logs: [] });
  const txt = fs.readFileSync(logsPath,'utf8');
  const lines = txt.trim().split(/\n+/).filter(Boolean).map(l=>JSON.parse(l));
  res.json({ logs: lines.reverse().slice(0,200) });
});

const PORT = process.env.PORT || 3000;
app.listen(PORT, ()=>console.log(`License Admin server (per-domain keys) running on :${PORT}`));
